
public final class CubeReader4 {
  // textual representation of centers
  private static final String[] midString = {
    "U", "D", "F", "B", "L", "R",
  };
  // textual representation of edges
  private static final String[] edgeString = {
    "UF", "UR", "UB", "UL", "DF", "DR", "DB", "DL", "FR", "FL", "BR", "BL",
    "FU", "RU", "BU", "LU", "FD", "RD", "BD", "LD", "RF", "LF", "RB", "LB"
  };
  // textual representation of corners
  private static final String[] cornString = {
    "UFR", "URB", "UBL", "ULF", "DRF", "DFL", "DLB", "DBR",
    "FRU", "RBU", "BLU", "LFU", "RFD", "FLD", "LBD", "BRD",
    "RUF", "BUR", "LUB", "FUL", "FDR", "LDF", "BDL", "RDB",
    // orientation is determined just by the first character
    "URF", "UBR", "ULB", "UFL", "DFR", "DLF", "DBL", "DRB",
    "FUR", "RUB", "BUL", "LUF", "RDF", "FDL", "LDB", "BDR",
    "RFU", "BRU", "LBU", "FLU", "FRD", "LFD", "BLD", "RBD"
  };
  private static final int[] revbits = {0, 4, 2, 6, 1, 5, 3, 7};
  private static final int[] edgeor = {0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 0, 1};

  private int[] e = new int[24]; // locations of the edges
  private int[] ep1 = new int[24]; // mark of locations of the edges
  private int[] c = new int[7]; // locations and orientations of the corners
  private int[] cp = new int[7]; // locations of the corners
  private int[] cp1 = new int[7]; // mark of locations of the corners
  private int[] co = new int[7]; // orientations of the corners
  private int[] co1 = new int[7]; // mark of orientations of the corners
  private int[] m = new int[24]; // locations of centers
  private int[] mp1 = new int[24]; // mark of locations of the centers
  private Solution4 solution;
  private CubeState4 state; // characteristic values of the cube state

  public String init(String input) {
    state = new CubeState4();
    String r = enter(input);
    if (r != "OK")
      return r;
    preprocess();
    return "OK";
  }

  public void solve() {
    System.gc();
    TurnList4 turnList = new TurnList4(200000, state.turnMask);
    System.gc();
    Transform4 transform = new Transform4(state);
    Prune4 prune = new Prune4(transform, state);
    solution = new Solution4(transform, prune, turnList);
    solution.solve(state);
  }

  private String enter(String input) {
    // read input from a user
    this.input = input;
    inputPos = 0;
    state.turnMask = 0777777;
    // reading the textual representation of 24 edges
    String[] edgeStr = new String[24];
    for (int i = 0; i < 24; i++) {
      edgeStr[i] = getInputToken();
      if (edgeStr[i].equals("Q") || edgeStr[i].equals("q"))
        return "QUIT";
      if (edgeStr[i].equals("N") || edgeStr[i].equals("n"))
        return "NEW";
      if (edgeStr[i].equals(""))
        return "Insufficient number of edges";
      if (edgeStr[i].charAt(0) >= '0' && edgeStr[i].charAt(0) <= '7') {
        state.turnMask = state.turnMask & 0777770 | revbits[edgeStr[i].charAt(0) - '0'];
        if (edgeStr[i].length() > 1 && edgeStr[i].charAt(1) >= '0' && edgeStr[i].charAt(1) <= '7')
          state.turnMask = state.turnMask & 0777707 | revbits[edgeStr[i].charAt(1) - '0'] << 3;
        if (edgeStr[i].length() > 2 && edgeStr[i].charAt(2) >= '0' && edgeStr[i].charAt(2) <= '7')
          state.turnMask = state.turnMask & 0777077 | revbits[edgeStr[i].charAt(2) - '0'] << 6;
        if (edgeStr[i].length() > 3 && edgeStr[i].charAt(3) >= '0' && edgeStr[i].charAt(3) <= '7')
          state.turnMask = state.turnMask & 0770777 | revbits[edgeStr[i].charAt(3) - '0'] << 9;
        if (edgeStr[i].length() > 4 && edgeStr[i].charAt(4) >= '0' && edgeStr[i].charAt(4) <= '7')
          state.turnMask = state.turnMask & 0707777 | revbits[edgeStr[i].charAt(4) - '0'] << 12;
        if (edgeStr[i].length() > 5 && edgeStr[i].charAt(5) >= '0' && edgeStr[i].charAt(5) <= '7')
          state.turnMask = state.turnMask & 0077777 | revbits[edgeStr[i].charAt(5) - '0'] << 15;
        i--;
      }
      else if (edgeStr[i].equals("!!"))
        for (; i < 24; i++)
          edgeStr[i] = "!";
      else if (edgeStr[i].equals("??"))
        for (; i < 24; i++)
          edgeStr[i] = "?";
    }
    // reading the textual representation of 7 corners (DBR is fixed)
    String[] cornStr = new String[7];
    for (int i = 0; i < 7; i++) {
      cornStr[i] = getInputToken();
      if (cornStr[i].equals("Q") || cornStr[i].equals("q"))
        return "QUIT";
      if (cornStr[i].equals("N") || cornStr[i].equals("n"))
        return "NEW";
      if (cornStr[i].equals(""))
        return "Insufficient number of corners";
      if (cornStr[i].equals("!!"))
        for (; i < 7; i++)
          cornStr[i] = "!";
      else if (cornStr[i].equals("@!!"))
        for (; i < 7; i++)
          cornStr[i] = "@!";
      else if (cornStr[i].equals("??"))
        for (; i < 7; i++)
          cornStr[i] = "?";
      else if (cornStr[i].equals("@??"))
        for (; i < 7; i++)
          cornStr[i] = "@?";
    }
    // reading the textual representation of 24 centers
    String[] midStr = new String[24];
    for (int i = 0; i < 24; i++) {
      midStr[i] = getInputToken();
      if (midStr[i].equals("Q") || midStr[i].equals("q"))
        return "QUIT";
      if (midStr[i].equals("N") || midStr[i].equals("n"))
        return "NEW";
      if (midStr[i].equals(""))
        return "Insufficient number of edges";
      if (midStr[i].equals("!!"))
        for (; i < 24; i++)
          midStr[i] = "!";
      else if (midStr[i].equals("??"))
        for (; i < 24; i++)
          midStr[i] = "?";
    }
    // filling the edge locations and orientations to e[24], ep1[24]
    for (int i = 0; i < 24; i++) {
      e[i] = 0;
      ep1[i] = 1;
    }
    for (int i = 0; i < 24; i++) {
      if (edgeStr[i].equals("!")) {
        e[i] = i + 1;
        ep1[i] = 0;
      }
      else if (edgeStr[i].equals("?")) {
      }
      else {
        boolean found = false;
        for (int j = 0; j < edgeString.length; j++) {
          if (edgeStr[i].equals(edgeString[j])) {
            int or = (i & 1) ^ (j / 12) ^ (edgeor[i / 2] ^ edgeor[j % 12]); // position in the pair
            e[i] = j % 12 * 2 + or + 1; // compute the real edge in the position
            ep1[e[i] - 1] = 0;
            found = true;
            break;
          }
        }
        if (!found)
          return "Improper edge: " + edgeStr[i];
      }
    }
    // filling the corner locations and orientations to c[7], co*[7], cp*[7]
    // co1[] is not completely filled (we do not know which corner is '@?')
    // we fill locations that are not set later
    for (int i = 0; i < 7; i++) {
      c[i] = 0;
      cp[i] = 0;
      co[i] = 0;
      cp1[i] = 1; // ok for unknown
      co1[i] = -1; // fixed later for unknown
    }
    for (int i = 0; i < 7; i++) {
      if (cornStr[i].equals("!")) {
        c[i] = i + 1;
        cp[i] = i + 1;
        co[i] = 1;
        cp1[i] = 0;
        co1[i] = 0;
      }
      else if (cornStr[i].equals("@!")) {
        c[i] = i + 1;
        cp[i] = i + 1;
        cp1[i] = 0;
        co1[i] = 1;
      }
      else if (cornStr[i].equals("+!")) {
        c[i] = i + 8 + 1;
        cp[i] = i + 1;
        co[i] = 2;
        cp1[i] = 0;
        co1[i] = 0;
      }
      else if (cornStr[i].equals("-!")) {
        c[i] = i + 16 + 1;
        cp[i] = i + 1;
        co[i] = 3;
        cp1[i] = 0;
        co1[i] = 0;
      }
      else if (cornStr[i].equals("?")) {
        co[i] = 1;
      }
      else if (cornStr[i].equals("@?")) {
      }
      else if (cornStr[i].equals("+?")) {
        co[i] = 2;
      }
      else if (cornStr[i].equals("-?")) {
        co[i] = 3;
      }
      else if (cornStr[i].charAt(0) == '@') {
        boolean found = false;
        for (int j = 0; j < cornString.length; j++) {
          if (cornStr[i].substring(1).equals(cornString[j])) {
            if (j % 8 == 7)
              return "Fixed corner: " + cornStr[i];
            c[i] = j % 24 % 8 + 1;
            cp[i] = (c[i] - 1) % 8 + 1;
            cp1[cp[i] - 1] = 0;
            co1[cp[i] - 1] = 1;
            found = true;
            break;
          }
        }
        if (!found)
          return "Improper corner: " + cornStr[i];
      }
      else if (cornStr[i].charAt(0) == '+') {
        boolean found = false;
        for (int j = 0; j < cornString.length; j++) {
          if (cornStr[i].substring(1).equals(cornString[j])) {
            if (j % 8 == 7)
              return "Fixed corner: " + cornStr[i];
            c[i] = (j + 8) % 24 + 1;
            cp[i] = (c[i] - 1) % 8 + 1;
            co[i] = (c[i] - 1) / 8 + 1;
            cp1[cp[i] - 1] = 0;
            co1[cp[i] - 1] = 0;
            found = true;
            break;
          }
        }
        if (!found)
          return "Improper corner: " + cornStr[i];
      }
      else if (cornStr[i].charAt(0) == '-') {
        boolean found = false;
        for (int j = 0; j < cornString.length; j++) {
          if (cornStr[i].substring(1).equals(cornString[j])) {
            if (j % 8 == 7)
              return "Fixed corner: " + cornStr[i];
            c[i] = (j + 16) % 24 + 1;
            cp[i] = (c[i] - 1) % 8 + 1;
            co[i] = (c[i] - 1) / 8 + 1;
            cp1[cp[i] - 1] = 0;
            co1[cp[i] - 1] = 0;
            found = true;
            break;
          }
        }
        if (!found)
          return "Improper corner: " + cornStr[i];
      }
      else {
        boolean found = false;
        for (int j = 0; j < cornString.length; j++) {
          if (cornStr[i].equals(cornString[j])) {
            if (j % 8 == 7)
              return "Fixed corner: " + cornStr[i];
            c[i] = j % 24 + 1;
            cp[i] = (c[i] - 1) % 8 + 1;
            co[i] = (c[i] - 1) / 8 + 1;
            cp1[cp[i] - 1] = 0;
            co1[cp[i] - 1] = 0;
            found = true;
            break;
          }
        }
        if (!found)
          return "Improper corner: " + cornStr[i];
      }
    }
    // fix the rest of co1[]
    // it is not really correct, since it represents only one of many states
    // however, it is handled correctly in the solving algorithm
    for (int i = 0; i < 7; i++) {
      if (cp[i] == 0) {  // where we didn't know the index to the co1[]
        if (co[i] != 0) { // and the corner is oriented
          boolean found = false;
          for (int j = 0; j < 7; j++) {
            if (co1[j] == -1) {
              co1[j] = 0;
              found = true;
              break;
            }
          }
          if (!found)
            return "Bad corner format";
        }
        else { // where the corner is not oriented
          boolean found = false;
          for (int j = 0; j < 7; j++) {
            if (co1[j] == -1) {
              co1[j] = 1;
              found = true;
              break;
            }
          }
          if (!found)
            return "Bad corner format";
        }
      }
    }
    for (int i = 0; i < 7; i++)
      if (co1[i] == -1)
        return "Bad corner format";
    // filling the center locations and orientations to m[24]
    for (int i = 0; i < 24; i++) {
      m[i] = 0;
    }
    for (int i = 0; i < 24; i++) {
      if (midStr[i].equals("!")) {
        m[i] = i / 4 + 1;
      }
      else if (midStr[i].equals("?")) {
      }
      else {
        boolean found = false;
        for (int j = 0; j < midString.length; j++) {
          if (midStr[i].equals(midString[j])) {
            m[i] = j + 1;
            found = true;
            break;
          }
        }
        if (!found)
          return "Improper center: " + midStr[i];
      }
    }
    int[] mids = new int[6];
    for (int i = 0; i < 24; i++)
      mp1[i] = 1;
    for (int i = 0; i < 24; i++) {
      if (m[i] > 0) {
        int j = mids[m[i] - 1]++;
        mp1[(m[i] - 1) * 4 + j] = 0;
      }
    }
    for (int i = 0; i < 6; i++) {
      if (mids[i] > 4)
        return mids[i] + " " + midString[i] + " centers";
    }
    return check();
  }

  private String check() {
    Comb edgeComb = new Comb(24);
    edgeComb.fill(e);
    if (!edgeComb.checkVar())
      return "Bad edges";
    Comb cornComb = new Comb(7);
    cornComb.fill(cp);
    if (!cornComb.checkVar())
      return "Bad corners";
    int cornParity = cornComb.varParity(); // parity of corner permutation (0, 1, -1, -2)
    int edgeParity = edgeComb.varParity(); // parity of edge permutation (0, 1, -1, -2)
    // for the fully specified state the parities must be the same
    // else must not depend on positions of min. 2 cubies of the same type
    if (cornParity == -1)
      return "Specify the position of the last corner";
    if (edgeParity == -1)
      return "Specify the position of the last edge";
    int twist = 0;
    for (int i = 0; i < 7; i++) { // number of clockwise twists of corners
      if (co[i] == 0) {
        twist = -1;
        break;
      }
      twist += co[i] - 1;
    }
    if (twist >= 0 && twist % 3 != 0) // twist mod 3 must be 0
      return "Twist any corner " + (twist % 3 == 1 ? "counter " : "") + "clock-wise";
    return "OK";
  }

  private void preprocess() {
    state.minimum = 101;
    state.depth = 0;
    // print states
    System.err.println("Edge:");
    for (int i = 0; i < 24; i++)
      System.err.print(e[i] == 0 ? "  ." : padInt(e[i] - 1, 3));
    System.err.println();
    for (int i = 0; i < 24; i++)
      System.err.print("  " + (ep1[i] == 0 ? "#" : "?"));
    System.err.println();
    System.err.println("Corner:");
    for (int i = 0; i < 7; i++)
      System.err.print(c[i] == 0 ? "  ." : padInt(c[i] - 1, 3));
    System.err.println();
    for (int i = 0; i < 7; i++)
      System.err.print(cp[i] == 0 ? "  ." : padInt(cp[i] - 1, 3));
    System.err.println();
    for (int i = 0; i < 7; i++)
      System.err.print("  " + (cp1[i] == 0 ? "#" : "?"));
    System.err.println();
    for (int i = 0; i < 7; i++)
      System.err.print(co[i] == 0 ? "  ." : padInt(co[i] % 3 - 1, 3));
    System.err.println();
    for (int i = 0; i < 7; i++)
      System.err.print("  " + (co1[i] == 0 ? "#" : "?"));
    System.err.println();
    System.err.println("Center:");
    for (int i = 0; i < 24; i++)
      System.err.print(m[i] == 0 ? "  ." : padInt(m[i] - 1, 3));
    System.err.println();
    for (int i = 0; i < 24; i++)
      System.err.print("  " + (mp1[i] == 0 ? "#" : "?"));
    System.err.println();
    // corner processing
    int cornOri = 0;
    int cornPos = 0;
    for (int i = 0; i < 7; i++) {
      cornOri |= co1[i] << i;
      cornPos |= cp1[i] << i;
    }
    cornOri |= cornPos << 8; // we have to specify which oriented corners are not positioned
    state.coActive = cornOri;
    state.cActive = cornPos;
    // corner-orientation processing
    PackCornTwist4 cornTwist = new PackCornTwist4(cornOri);
    cornTwist.fill(co);
    state.cornTwist = cornTwist.pack();
    // corner-permutation processing
    PackCornPerm4 cornPerm = new PackCornPerm4(cornPos);
    cornPerm.fill(cp);
    state.cornPerm = cornPerm.pack();
    // edge processing
    int edgePos = 0;
    for (int i = 0; i < 24; i++)
      edgePos |= ep1[i] << i;
    state.eActive = edgePos;
    int[] et = new int[24];
    int r = 1; // UF
    for (int i = 0; i < 24; i++)
      et[i] = 0;
    for (int i = 0; i < 24; i++)
      if (e[i] == 1) et[i] = r++;
    for (int i = 0; i < 24; i++)
      if (e[i] == 2) et[i] = r;
    PackEdgePos4 edgePosUF = new PackEdgePos4(edgePos, 0);
    edgePosUF.fill(et);
    state.edgePosUF = edgePosUF.pack();
    r = 1; // UR
    for (int i = 0; i < 24; i++)
      et[i] = 0;
    for (int i = 0; i < 24; i++)
      if (e[i] == 3) et[i] = r++;
    for (int i = 0; i < 24; i++)
      if (e[i] == 4) et[i] = r;
    PackEdgePos4 edgePosUR = new PackEdgePos4(edgePos, 2);
    edgePosUR.fill(et);
    state.edgePosUR = edgePosUR.pack();
    r = 1; // UB
    for (int i = 0; i < 24; i++)
      et[i] = 0;
    for (int i = 0; i < 24; i++)
      if (e[i] == 5) et[i] = r++;
    for (int i = 0; i < 24; i++)
      if (e[i] == 6) et[i] = r;
    PackEdgePos4 edgePosUB = new PackEdgePos4(edgePos, 4);
    edgePosUB.fill(et);
    state.edgePosUB = edgePosUB.pack();
    r = 1; // UL
    for (int i = 0; i < 24; i++)
      et[i] = 0;
    for (int i = 0; i < 24; i++)
      if (e[i] == 7) et[i] = r++;
    for (int i = 0; i < 24; i++)
      if (e[i] == 8) et[i] = r;
    PackEdgePos4 edgePosUL = new PackEdgePos4(edgePos, 6);
    edgePosUL.fill(et);
    state.edgePosUL = edgePosUL.pack();
    r = 1; // DF
    for (int i = 0; i < 24; i++)
      et[i] = 0;
    for (int i = 0; i < 24; i++)
      if (e[i] == 9) et[i] = r++;
    for (int i = 0; i < 24; i++)
      if (e[i] == 10) et[i] = r;
    PackEdgePos4 edgePosDF = new PackEdgePos4(edgePos, 8);
    edgePosDF.fill(et);
    state.edgePosDF = edgePosDF.pack();
    r = 1; // DR
    for (int i = 0; i < 24; i++)
      et[i] = 0;
    for (int i = 0; i < 24; i++)
      if (e[i] == 11) et[i] = r++;
    for (int i = 0; i < 24; i++)
      if (e[i] == 12) et[i] = r;
    PackEdgePos4 edgePosDR = new PackEdgePos4(edgePos, 10);
    edgePosDR.fill(et);
    state.edgePosDR = edgePosDR.pack();
    r = 1; // DB
    for (int i = 0; i < 24; i++)
      et[i] = 0;
    for (int i = 0; i < 24; i++)
      if (e[i] == 13) et[i] = r++;
    for (int i = 0; i < 24; i++)
      if (e[i] == 14) et[i] = r;
    PackEdgePos4 edgePosDB = new PackEdgePos4(edgePos, 12);
    edgePosDB.fill(et);
    state.edgePosDB = edgePosDB.pack();
    r = 1; // DL
    for (int i = 0; i < 24; i++)
      et[i] = 0;
    for (int i = 0; i < 24; i++)
      if (e[i] == 15) et[i] = r++;
    for (int i = 0; i < 24; i++)
      if (e[i] == 16) et[i] = r;
    PackEdgePos4 edgePosDL = new PackEdgePos4(edgePos, 14);
    edgePosDL.fill(et);
    state.edgePosDL = edgePosDL.pack();
    r = 1; // FR
    for (int i = 0; i < 24; i++)
      et[i] = 0;
    for (int i = 0; i < 24; i++)
      if (e[i] == 17) et[i] = r++;
    for (int i = 0; i < 24; i++)
      if (e[i] == 18) et[i] = r;
    PackEdgePos4 edgePosFR = new PackEdgePos4(edgePos, 16);
    edgePosFR.fill(et);
    state.edgePosFR = edgePosFR.pack();
    r = 1; // FL
    for (int i = 0; i < 24; i++)
      et[i] = 0;
    for (int i = 0; i < 24; i++)
      if (e[i] == 19) et[i] = r++;
    for (int i = 0; i < 24; i++)
      if (e[i] == 20) et[i] = r;
    PackEdgePos4 edgePosFL = new PackEdgePos4(edgePos, 18);
    edgePosFL.fill(et);
    state.edgePosFL = edgePosFL.pack();
    r = 1; // BR
    for (int i = 0; i < 24; i++)
      et[i] = 0;
    for (int i = 0; i < 24; i++)
      if (e[i] == 21) et[i] = r++;
    for (int i = 0; i < 24; i++)
      if (e[i] == 22) et[i] = r;
    PackEdgePos4 edgePosBR = new PackEdgePos4(edgePos, 20);
    edgePosBR.fill(et);
    state.edgePosBR = edgePosBR.pack();
    r = 1; // BL
    for (int i = 0; i < 24; i++)
      et[i] = 0;
    for (int i = 0; i < 24; i++)
      if (e[i] == 23) et[i] = r++;
    for (int i = 0; i < 24; i++)
      if (e[i] == 24) et[i] = r;
    PackEdgePos4 edgePosBL = new PackEdgePos4(edgePos, 22);
    edgePosBL.fill(et);
    state.edgePosBL = edgePosBL.pack();
    // center processing
    int midPos = 0;
    for (int i = 0; i < 24; i++)
      midPos |= mp1[i] << i;
    state.mActive = midPos;
    int[] mt = new int[24];
    for (int i = 0; i < 24; i++) // U
      mt[i] = m[i] == 1 ? 1 : 0;
    PackMidLoc4 midLocU = new PackMidLoc4(midPos, 0);
    midLocU.fill(mt);
    state.midLocU = midLocU.pack();
    for (int i = 0; i < 24; i++) // D
      mt[i] = m[i] == 2 ? 1 : 0;
    PackMidLoc4 midLocD = new PackMidLoc4(midPos, 4);
    midLocD.fill(mt);
    state.midLocD = midLocD.pack();
    for (int i = 0; i < 24; i++) // F
      mt[i] = m[i] == 3 ? 1 : 0;
    PackMidLoc4 midLocF = new PackMidLoc4(midPos, 8);
    midLocF.fill(mt);
    state.midLocF = midLocF.pack();
    for (int i = 0; i < 24; i++) // B
      mt[i] = m[i] == 4 ? 1 : 0;
    PackMidLoc4 midLocB = new PackMidLoc4(midPos, 12);
    midLocB.fill(mt);
    state.midLocB = midLocB.pack();
    for (int i = 0; i < 24; i++) // L
      mt[i] = m[i] == 5 ? 1 : 0;
    PackMidLoc4 midLocL = new PackMidLoc4(midPos, 16);
    midLocL.fill(mt);
    state.midLocL = midLocL.pack();
    for (int i = 0; i < 24; i++) // R
      mt[i] = m[i] == 6 ? 1 : 0;
    PackMidLoc4 midLocR = new PackMidLoc4(midPos, 20);
    midLocR.fill(mt);
    state.midLocR = midLocR.pack();
    // additional corner-permutation processing
    int[] tcp = new int[7];
    for (int i = 0; i < 7; i++) {
      if (cp[i] == 1 || cp[i] == 3 || cp[i] == 4 || cp[i] == 6)
        tcp[i] = cp[i];
      else
        tcp[i] = 0;
    }
    PackACornPerm4 aCornPerm = new PackACornPerm4(cornPos);
    aCornPerm.fill(tcp);
    state.aCornPerm = aCornPerm.pack();
    for (int i = 0; i < 7; i++) {
      if (cp[i] == 2 || cp[i] == 4 || cp[i] == 5 || cp[i] == 7)
        tcp[i] = cp[i];
      else
        tcp[i] = 0;
    }
    PackOCornPerm4 oCornPerm = new PackOCornPerm4(cornPos);
    oCornPerm.fill(tcp);
    state.oCornPerm = oCornPerm.pack();
  }

  private String input;
  private int inputPos;

  private String getInputToken() {
    String s = "";
    while (inputPos < input.length() && input.charAt(inputPos) <= 32)
      inputPos++;
    while (inputPos < input.length() && input.charAt(inputPos) > 32) {
      s += input.charAt(inputPos);
      inputPos++;
    }
    return s;
  }

  public static String padInt(int number, int length) {
    String s = "" + number;
    while (s.length() < length)
      s = " " + s;
    return s;
  }
}
